
/*****************************************************************************
 *  @file testfile.c
 *  @brief A simple coding framework.
 *
 *  @date   : 23-05-04 18:00
 *  @author : Pedro Ortega C. <peortega@dcc.uchile.cl>
 *  Copyright  2004  Pedro Ortega C.
 ****************************************************************************/
/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



/* Include Files */

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>

#include <libgnn.h>



/* Some macros and definitions. */

#define LIBGNN_MINARGS      5
#define LIBGNN_ARG_PROGNAME 0
#define LIBGNN_ARG_MODE     (LIBGNN_ARG_PROGNAME + 1)
#define LIBGNN_ARG_PARAMS   (LIBGNN_ARG_MODE     + 1)
#define LIBGNN_ARG_INPUTS   (LIBGNN_ARG_PARAMS   + 1)
#define LIBGNN_ARG_TARGETS  (LIBGNN_ARG_INPUTS   + 1)
#define LIBGNN_ARG_OUTPUTS  (LIBGNN_ARG_INPUTS   + 1)

#define LIBGNN_CMD_TRAIN "train"
#define LIBGNN_CMD_EVAL  "eval"



/* Function declaration. */

gnn_node*      libgnn_build_machine ();
gnn_criterion* libgnn_build_criterion ();
gnn_trainer*   libgnn_build_trainer (gnn_node      *machine,
                                     gnn_criterion *crit,
                                     gnn_dataset   *trainset);
int libgnn_trainloop (gnn_trainer *trainer, size_t maxEpochs);


int libgnn_help (int argc, const char **argv);
int libgnn_param_load (gnn_node *machine, const char *paramFile);
int libgnn_param_save (gnn_node *machine, const char *paramFile);
int libgnn_train (int argc, const char **argv);
int libgnn_eval (int argc, const char **argv);
int libgnn_main (int argc, const char **argv);
int main (int argc, const char **argv);



/* Implementation. */

/*****************************************************************************
 * YOU SHOULD FILL OUT THIS                                                  *
 *****************************************************************************/

/* The name of the executable. */
#define LIBGNN_PROGNAME "myexample"


/**
 * Two-Layer-Perceptron.
 * Arquitecture: 40-20-3.
 * Activation function: sigmoidal.
 */
gnn_node*
libgnn_build_machine ()
{
    gnn_node *w1;
    gnn_node *w2;
    gnn_node *s1;
    gnn_node *s2;
    gnn_node *nn;

    w1 = gnn_weight_new (40, 20);
    s1 = gnn_logistic_standard_new (20);
    w2 = gnn_weight_new (20, 3);
    s2 = gnn_logistic_standard_new (3);
    if (w1 == NULL || w2 == NULL || s1 == NULL || s2 == NULL)
    {
        fprintf (stderr, "Couldn't build one or more components "
                         "of the neural net.\n");
        return NULL;
    }

    gnn_weight_init (w1);
    gnn_weight_init (w2);

    nn = gnn_serial_new (4, w1, s1, w2, s2);
    if (nn == NULL)
    {
        fprintf (stderr, "Couldn't build neural net.\n");
        return NULL;
    }

    return nn;
}

/**
 * Cross-entropy works nice for classification problems.
 */
gnn_criterion*
libgnn_build_criterion ()
{
    // return gnn_mse_new (3);
    return gnn_cross_entropy_new (3);
}

/**
 * Conjugate-Gradients is ok.
 */
gnn_trainer*
libgnn_build_trainer (gnn_node *machine,
                      gnn_criterion *crit, gnn_dataset *trainset)
{
    // return gnn_rprop_standard_new (machine, crit, trainset);
    // return gnn_gradient_descent_new (machine, crit, trainset, 0.001);
    // return gnn_momentum_new (machine, crit, trainset, 0.001, 0.8);
    // return gnn_bfgs_new (machine, crit, trainset);
    // return gnn_lmbfgs_new (machine, crit, trainset);
    return gnn_conjugate_gradient_new (machine, crit, trainset);
}

/**
 * The framework calls this function in order to execute the train the
 * machine. You should modify this if you want to display extra info, etc.
 */
int
libgnn_trainloop (gnn_trainer *trainer, size_t maxEpochs)
{
    double error;
    size_t epoch;
    
    gnn_trainer_reset (trainer);
    epoch = gnn_trainer_get_epoch (trainer);
    
    for (; epoch <= maxEpochs; epoch = gnn_trainer_get_epoch (trainer))
    {
        gnn_trainer_train (trainer);
        error = gnn_trainer_get_epoch_cost (trainer);
        printf ("Epoch: %3d - Error : %10.6f \r", epoch, error);
    }
    printf ("\n");
    
    return 0;
}













/*****************************************************************************
 * THE FOLLOWING CODE SHOULDN'T BE MODIFIED, BUT READ IT, UNDERSTAND IT      *
 * AND USE IT AS A GUIDE FOR YOUR OWN PROGRAMS. IT'S QUITE SIMPLE.           *
 *****************************************************************************/


/**
 * @brief Print help message.
 *
 * This function prints a help message.
 *
 * @param argc Number of arguments.
 * @param argv String vector of arguments.
 */
int
libgnn_help (int argc, const char **argv)
{
    /* Print help message. */
    printf (
        "\n"
        "Use: %s train <grm> <inputs> <targets> [epochs] [batchsize]\n"
        "or\n"
        "Use: %s eval  <grm> <inputs> <outputs>\n"
        "\n"
        "where:\n"
        "   [epochs]    : Number of epochs. Default value is 100.\n"
        "   [batchsize] : Size of the training batches. Default: all.\n"
        "   <grm>       : Parameter file.\n"
        "   <inputs>    : File with inputs.\n"
        "   <targets>   : File with targets.\n"
        "   <outputs>   : File where output should be placed.\n"
        ,
        LIBGNN_PROGNAME,
        LIBGNN_PROGNAME
    );
    
    return 0;
}



/**
 * @brief Load parameters into a gnn_node.
 *
 * This function loads parameters from a given file into the given
 * \ref gnn_node. Returns 0 if OK, 1 if the file couldn't be opened,
 * and -1 if an error ocurred.
 *
 * The parameters should be in ASCII-Format.
 *
 * @param machine   A \ref gnn_node.
 * @param paramFile The filename of the parameters.
 */
int
libgnn_param_load (gnn_node *machine, const char *paramFile)
{
    FILE *fp = NULL;
    gsl_vector *w = NULL;
    gsl_vector_int *f = NULL;
    size_t paramSize = 0;

    /* Get number of parameters. */
    paramSize = gnn_node_param_get_size (machine);
    
    /* Open the file. */
    fp = fopen (paramFile, "r");
    if (fp == NULL)
    {
        /* File doesn't exist. */
        return 1;
    }

    /* File exists => parameters should be loaded. */
    
    /* Allocate a vector where to store the parameters. */
    w = gsl_vector_alloc (paramSize);
    if (w == NULL)
    {
        fprintf (stderr, "Couldn't allocate memory for parameter loading.\n");
        return -1;
    }
    
    f = gsl_vector_int_alloc (paramSize);
    if (f == NULL)
    {
        fprintf (stderr, "Couldn't allocate memory for parameter loading.\n");
        return -1;
    }

    /* Read parameter vector. */
    gsl_vector_fscanf (fp, w);
    gsl_vector_int_fscanf (fp, f);
    fclose (fp);

    /* Set parameters. */
    gnn_node_param_set (machine, w);
    gnn_node_param_freeze_flags_set (machine, f);

    /* Clean all. */
    gsl_vector_free (w);
    gsl_vector_int_free (f);

    return 0;
}



/**
 * @brief Save a gnn_node's parameters into text file.
 *
 * This function saves the given gnn_node's parameters into a text file.
 * Returns 0 if OK, 1 if the file couldn't be opened,
 * and -1 if an error ocurred.
 *
 * The parameters are stored in ASCII-Format and can be read again
 * using \ref libgnn_param_load.
 *
 * @param machine   A \ref gnn_node.
 * @param paramFile The filename of the parameters.
 */
int
libgnn_param_save (gnn_node *machine, const char *paramFile)
{
    FILE *fp = NULL;
    gsl_vector *w = NULL;
    gsl_vector_int *f = NULL;
    size_t paramSize = 0;

    /* Get number of parameters. */
    paramSize = gnn_node_param_get_size (machine);

    /* Open the file. */
    fp = fopen (paramFile, "w");
    if (fp == NULL)
    {
        /* File doesn't exist. */
        return 1;
    }

    /* Allocate a vector where to store the parameters. */
    w = gsl_vector_alloc (paramSize);
    if (w == NULL)
    {
        fprintf (stderr, "Couldn't allocate memory for parameter saving.\n");
        return -1;
    }
    
    f = gsl_vector_int_alloc (paramSize);
    if (f == NULL)
    {
        fprintf (stderr, "Couldn't allocate memory for parameter saving.\n");
        return -1;
    }

    /* Get parameters. */
    gnn_node_param_get (machine, w);
    gnn_node_param_freeze_flags_get (machine, f);

    /* Save parameter vector. */
    gsl_vector_fprintf (fp, w, "%g");
    gsl_vector_int_fprintf (fp, f, "%d");
    fclose (fp);

    /* Clean all. */
    gsl_vector_free (w);
    gsl_vector_int_free (f);

    return 0;
}



/**
 * @brief Train method.
 *
 * This function implements the whole Training-Process, i.e. initialization
 * of the gradient machine, criterion building, trainer building, data
 * loading, etc.
 *
 * The train method calls four important functions:
 * -# \ref libgnn_build_machine : Builds the GRM and returns the \ref gnn_node.
 * -# \ref libgnn_build_criterion : Builds the criterion.
 * -# \ref libgnn_build_trainer : Builds the trainer.
 * -# \ref libgnn_trainloop : The training loop.
 *
 * @param argc Number of arguments.
 * @param argv String vector of arguments.
 */
int
libgnn_train (int argc, const char **argv)
{
    int status = 0;
    
    gnn_dataset *trainset  = NULL;
    gnn_input   *inputs    = NULL;
    gnn_input   *targets   = NULL;
    
    gnn_node      *machine = NULL;
    gnn_criterion *crit    = NULL;
    gnn_trainer   *trainer = NULL;
    
    /* Custom train variables. */
    const char *inputFile  = NULL;
    const char *targetFile = NULL;
    const char *paramFile  = NULL;
    size_t batchSize;
    size_t maxEpochs;
    
    /* 1. Parse arguments. */
    printf ("Parse Arguments.\n");
    inputFile  = argv[LIBGNN_ARG_INPUTS];
    targetFile = argv[LIBGNN_ARG_TARGETS];
    paramFile  = argv[LIBGNN_ARG_PARAMS];
    
    batchSize = 0;
    if (argc >= 7)
    {
        batchSize  = atol (argv[6]);
    }
    
    maxEpochs = 100;
    if (argc >= 6)
    {
        maxEpochs = atol (argv[5]);
    }
    
    /* 2. Build machine. */
    printf ("Build machine.\n");
    machine = libgnn_build_machine ();
    if (machine == NULL)
    {
        fprintf (stderr, "Couldn't build machine.\n");
        return -1;
    }
    
    /* 3. Load parameters if file exists. */
    printf ("Load parameters if available.\n");
    status = libgnn_param_load (machine, paramFile);
    if (status < 0)
    {
        gnn_node_destroy (machine);
        return -1;
    }
    
    /* 4. Get train data. */
    printf ("Load train data.\n");
    inputs  = gnn_memory_input_new_from_file (inputFile);
    if (inputs == NULL)
    {
        fprintf (stderr, "Couldn't get inputs from \"%s\".\n", inputFile);
        gnn_node_destroy (machine);
        return -1;
    }

    targets = gnn_memory_input_new_from_file (targetFile);
    if (targets == NULL)
    {
        fprintf (stderr, "Couldn't get targets from \"%s\".\n", targetFile);
        gnn_node_destroy (machine);
        gnn_input_destroy (inputs);
        return -1;
    }

    trainset = gnn_simple_set_new (inputs, targets, NULL);
    if (trainset == NULL)
    {
        fprintf (stderr, "Couldn't create dataset.\n", targetFile);
        gnn_node_destroy (machine);
        gnn_input_destroy (inputs);
        gnn_input_destroy (targets);
        return -1;
    }

    /* 5. Build criterion. */
    printf ("Build criterion.\n");
    crit = libgnn_build_criterion ();
    if (crit == NULL)
    {
        fprintf (stderr, "Couldn't build criterion.\n");
        gnn_node_destroy (machine);
        gnn_dataset_destroy (trainset);
        return -1;
    }

    /* 6. Build trainer. */
    printf ("Build trainer.\n");
    trainer = libgnn_build_trainer (machine, crit, trainset);
    if (trainer == NULL)
    {
        fprintf (stderr, "Couldn't build trainer.\n");
        gnn_node_destroy (machine);
        gnn_dataset_destroy (trainset);
        gnn_criterion_destroy (crit);
        return -1;
    }
    
    if (batchSize == 0)
    {
        batchSize = gnn_dataset_get_size (trainset);
        gnn_trainer_batch_set_size (trainer, batchSize);
    }

    /* 7. Train. */
    printf ("Training.\n");
    status = libgnn_trainloop (trainer, maxEpochs);
    if (status != 0)
    {
        fprintf (stderr, "An error ocurred during training.\n");
        gnn_node_destroy (machine);
        gnn_criterion_destroy (crit);
        gnn_dataset_destroy (trainset);
        gnn_trainer_destroy (trainer);
        return -1;
    }

    /* 7. Save parameters. */
    printf ("Save parameters.\n");
    status = libgnn_param_save (machine, paramFile);
    if (status != 0)
    {
        fprintf (stderr, "Couldn't save the parameters into \"%s\".\n",
                         paramFile);
        return -1;
    }
    
    printf ("Free resources.\n");
    gnn_trainer_destroy (trainer);
    gnn_node_destroy (machine);
    gnn_criterion_destroy (crit);
    gnn_dataset_destroy (trainset);

    return 0;
}



/**
 * @brief Train method.
 *
 * This function implements the whole Training-Process, i.e. initialization
 * of the gradient machine, criterion building, trainer building, data
 * loading, etc.
 *
 * The train method calls four important functions:
 * -# \ref libgnn_build_machine : Builds the GRM and returns the \ref gnn_node.
 * -# \ref libgnn_build_criterion : Builds the criterion.
 * -# \ref libgnn_build_trainer : Builds the trainer.
 * -# \ref libgnn_trainloop : The training loop.
 *
 * @param argc Number of arguments.
 * @param argv String vector of arguments.
 */
int
libgnn_eval (int argc, const char **argv)
{
    int status = 0;

    gnn_input  *inputs    = NULL;
    gnn_output *outputs   = NULL;

    gnn_node *machine = NULL;
    gnn_eval *eval    = NULL;
    
    /* Custom train variables. */
    const char *inputFile  = NULL;
    const char *outputFile = NULL;
    const char *paramFile  = NULL;

    /* 1. Parse arguments. */
    inputFile  = argv[LIBGNN_ARG_INPUTS];
    outputFile = argv[LIBGNN_ARG_OUTPUTS];
    paramFile  = argv[LIBGNN_ARG_PARAMS];


    /* 2. Build machine. */
    machine = libgnn_build_machine ();
    if (machine == NULL)
    {
        fprintf (stderr, "Couldn't build machine.\n");
        return -1;
    }

    /* 3. Load parameters if file exists. */
    status = libgnn_param_load (machine, paramFile);
    if (status < 0)
    {
        gnn_node_destroy (machine);
        return -1;
    }

    /* 4. Get train data. */
    inputs  = gnn_memory_input_new_from_file (inputFile);
    if (inputs == NULL)
    {
        fprintf (stderr, "Couldn't get inputs from \"%s\".\n", inputFile);
        gnn_node_destroy (machine);
        return -1;
    }

    outputs = gnn_filewriter_with_file_new (outputFile);
    if (outputs == NULL)
    {
        fprintf (stderr, "Couldn't open output file \"%s\".\n", outputFile);
        gnn_node_destroy (machine);
        gnn_input_destroy (inputs);
        return -1;
    }
    gnn_filewriter_set_format (outputs, "%f\t");

    /* 5. Build evaluator and evaluate. */
    eval = gnn_eval_new (machine, inputs, outputs);
    if (eval == NULL)
    {
        fprintf (stderr, "Couldn't build evaluator.\n");
        gnn_node_destroy (machine);
        gnn_input_destroy (inputs);
        gnn_output_destroy (outputs);
        return -1;
    }
    
    gnn_eval_all (eval);

    gnn_node_destroy (machine);
    gnn_eval_destroy (eval);
    gnn_input_destroy (inputs);
    gnn_output_destroy (outputs);

    return 0;
}



/**
 * @brief Main method.
 *
 * This function corresponds to a normal main C function.
 *
 * @param argc Number of arguments.
 * @param argv String vector of arguments.
 */
int
libgnn_main (int argc, const char **argv)
{
    int status = 0;
    
    /* Check minimum argument number. */
    if (argc < LIBGNN_MINARGS)
    {
        libgnn_help (argc, argv);
        return -1;
    }

    /* Check execution mode. */
    if (strcmp (argv[LIBGNN_ARG_MODE], LIBGNN_CMD_TRAIN) == 0)
    {
        status = libgnn_train (argc, argv);
    }
    else if (strcmp (argv[LIBGNN_ARG_MODE], LIBGNN_CMD_EVAL) == 0)
    {
        status = libgnn_eval (argc, argv);
    }
    else
    {
        libgnn_help (argc, argv);
        return -1;
    }
    
    return status;
}



int
main (int argc, const char **argv)
{
    int status;
    
    status = libgnn_main (argc, argv);

    return status;
}
